/*
 *  SVD.c
 *  Matrix Plugin
 *
 *  Created by Robert Delaney on 12/12/09.
 *  Copyright 2009 Bob Delaney's Science Software. All rights reserved.
 *
 */

#include "SVD.h"
#include "diags.h"
#include <stdlib.h>
#include <math.h>

// orders D in decreasing order and applies resulting orthogonal transformation to V
// if switch i & j diagonal elements of D you switch columns i and j of V negating one of the columns
static void orderD(double *D, double **V, long n)
{
	double		*D1, **V1, temp;
	long		i, j, k, iSave, *S, save, sign;
	
	D1 = (double *)malloc(n*sizeof(double));
	
	V1 = (double **)malloc(n*sizeof(double *));
	for (i=0;i<n;i++)
		V1[i] = (double *)malloc(n*sizeof(double));
	
	for(i=0;i<n;i++)
	{
		D1[i] = D[i];
		for(j=0;j<n;j++)
			V1[i][j] = V[i][j];
	}
	
	// S will contain a description of the final sort
	S = (long *)malloc(n*sizeof(long));
	for(i=0;i<n;i++)
		S[i] = i+1;
	
	k = 0; // first D1 index to be switched with largest value
	iSave = 0; // avoid warning
	while(k<n-1)
	{
		temp = D1[k];
		iSave = k;
		for(i=k+1;i<n;i++)
		{
			if(D1[i]>temp)
			{
				temp = D1[i];
				iSave = i;
			}
		}
		if(iSave>k)
		{
			// swap
			D1[iSave] = D1[k];
			D[k] = temp;  // not needed in D1 any more
			// reflect this in S
			save = S[iSave];
			S[iSave] = -S[k];
			S[k] = save;
		}
		
		k++;
	}
	D[n-1] = D1[n-1];
	
	
	// now use S to arrange the columns of V
	for(j=0;j<n;j++)
	{
		// fill the jth column with the kth column pointed to by S[j]
		k = abs(S[j]) - 1;
		if(S[j]>0)
			sign = 1;
		else
			sign = -1;
		
		for(i=0;i<n;i++)
			V[i][j] = sign*V1[i][k];

	}
	
	free(D1);
	
	for (i=0;i<n;i++)
		free(V1[i]);
	free(V1);
	
}/* orderD */

// V will diagonalize M' * M
// give this m , n, and M (m x n), where m >= n
// it returns U (m x n), V (n x n), and the vector D (n x 1) of ordered singular values
// for the SVD (with D now a square diagonal matrix (n x n) with the singular values along its diagonal:
// M = U * D * V'   where V' is the transpose of V
// All matrices should be dynamically allocated by the calling program.
void svd(double **U, double *D, double **V, double **M, long m, long n, double eps)
{
	long		i, j, k;
	double		**A, **W, *d;
	
	A = (double **)malloc(n*sizeof(double *));
	for (i=0;i<n;i++)
		A[i] = (double *)malloc(n*sizeof(double));
	
	// A(n x n) = M' * M  A_ik = Sum(j=1 to m) M_j i * M_j k   i,k = 1, 2, ... , n
	for(i=0;i<n;i++)
		for(k=0;k<n;k++)
		{
			A[i][k] = 0;
			for(j=0;j<m;j++)
				A[i][k] = A[i][k] + M[j][i] * M[j][k];
		}
	
	// diagonalize A with V to hold eigenvectors
	diags(A, V, n, eps);  // the columns of V are reordered so the later D diagonal is ordered from high to low values
	
	// A is diagonal with the eigenvalues
	for(i=0;i<n;i++)
		D[i] = A[i][i];
	
	orderD(D, V, n);
	
	// form U = M * V
	// U_i k = Sum(j=1 to n) M_i j * V_j k    i = 1 , 2, ... , m   k = 1, 2, ... , n
	
	for(i=0;i<m;i++)
		for(k=0;k<n;k++)
		{
			U[i][k] = 0;
			for(j=0;j<n;j++)
				U[i][k] = U[i][k] + M[i][j] * V[j][k];
		}
	
	// the columns of U are orthogonal vectors except for null vectors, their norms are the singular values
	
	for(j=0;j<n;j++)
	{
		D[j] = 0;
		for(i=0;i<m;i++)
			D[j] = D[j] + U[i][j]*U[i][j];
		D[j] = sqrt(D[j]);
	}
	
	// divide each column by its norm unless the norm is zero
	for(j=0;j<n;j++)
	{
		for(i=0;i<m;i++)
		{
			if(D[j])
				U[i][j] = U[i][j] / D[j];
		}
	}
	
	// for any D = 0, we need to diagonalize M * M' with W and use the column eigenvectors with zero eigenvalues and put them into U
	
	if(D[n-1]*D[n-1]<eps)
	{
		for (i=0;i<n;i++)
			free(A[i]);
		free(A);
		
		A = (double **)malloc(m*sizeof(double *));
		for (i=0;i<m;i++)
			A[i] = (double *)malloc(m*sizeof(double));
		
		W = (double **)malloc(m*sizeof(double *));
		for (i=0;i<m;i++)
			W[i] = (double *)malloc(m*sizeof(double));
		
		d = (double *)malloc(m*sizeof(double));
		
		// A(m x m) = M * M'  A_ik = Sum(j=1 to n) M_i j * M_k j   i,k = 1, 2, ... , m
		for(i=0;i<m;i++)
			for(k=0;k<m;k++)
			{
				A[i][k] = 0;
				for(j=0;j<n;j++)
					A[i][k] = A[i][k] + M[i][j] * M[k][j];
			}
		
		// diagonalize A with W to hold eigenvectors
		diags(A, W, m, eps);
		
		//put eigenvalues into d
		for(i=0;i<m;i++)
			d[i] = A[i][i];
		
		orderD(d, W, m);
		
		// find the indices of the zero eigenvalues, and copy those columns of W to U
		for(j=0;j<n;j++)
		{
			if(d[j]<eps)
			{
				for(i=0;i<m;i++)
					U[i][j] = W[i][j];
			}
		}
		
		for (i=0;i<m;i++)
			free(A[i]);
		free(A);
		
		for (i=0;i<m;i++)
			free(W[i]);
		free(W);
		
		free(d);
		
	}
	
	for (i=0;i<n;i++)
		free(A[i]);
	free(A);
	
}/* svd */


// V will diagonalize M' * M
// give this m , n, V (n x n), and M (m x n), where m >= n
// it returns orthogonal U (m x m), V (n x n), and the vector D (n x 1) of ordered singular values
// for the SVD (with D now a pseudo-diagonal matrix (m x n)) with the singular values in upper diagonal n x n matrix and an (m-n) x n null matrix below:
// M = U * D * V'   where V' is the transpose of V
// All matrices should be dynamically allocated by the calling program.
void svd2(double **U, double *D, double **V, double **M, long m, long n, double eps)
{
	long		i, j, k;
	double		**A, **U1, *d;
	
	A = (double **)malloc(n*sizeof(double *));
	for (i=0;i<n;i++)
		A[i] = (double *)malloc(n*sizeof(double));
	
	// A(n x n) = M' * M  A_ik = Sum(j=1 to m) M_j i * M_j k   i,k = 1, 2, ... , n
	for(i=0;i<n;i++)
		for(k=0;k<n;k++)
		{
			A[i][k] = 0;
			for(j=0;j<m;j++)
				A[i][k] = A[i][k] + M[j][i] * M[j][k];
		}
	
	// diagonalize A with V to hold eigenvectors
	diags(A, V, n, eps);
	
	// A is diagonal with the eigenvalues
	for(i=0;i<n;i++)
		D[i] = A[i][i];
	
	orderD(D, V, n);  // the columns of V are reordered so the later D diagonal is ordered from high to low values
	
	U1 = (double **)malloc(m*sizeof(double *));
	for (i=0;i<m;i++)
		U1[i] = (double *)malloc(n*sizeof(double));
	
	// form U1 = M * V
	// U1_i k = Sum(j=1 to n) M_i j * V_j k    i = 1 , 2, ... , m   k = 1, 2, ... , n
	
	for(i=0;i<m;i++)
		for(k=0;k<n;k++)
		{
			U1[i][k] = 0;
			for(j=0;j<n;j++)
				U1[i][k] = U1[i][k] + M[i][j] * V[j][k];
		}
	
	// the columns of U1 are orthogonal vectors except for null vectors, their norms are the singular values
	
	for(j=0;j<n;j++)
	{
		D[j] = 0;
		for(i=0;i<m;i++)
			D[j] = D[j] + U1[i][j]*U1[i][j];
		D[j] = sqrt(D[j]);
	}
	
	// divide each column by its norm unless the norm is zero
	for(j=0;j<n;j++)
	{
		for(i=0;i<m;i++)
		{
			if(D[j])
				U1[i][j] = U1[i][j] / D[j];
		}
	}
	
	// we need to diagonalize M * M' with U and use the column eigenvectors with zero eigenvalues and put them as last columns of U with the leading columns being those of U1 until we reach "null" vectors (norm^2 < eps)
	for (i=0;i<n;i++)
		free(A[i]);
	free(A);
	
	A = (double **)malloc(m*sizeof(double *));
	for (i=0;i<m;i++)
		A[i] = (double *)malloc(m*sizeof(double));
	
	d = (double *)malloc(m*sizeof(double));
	
	// A(m x m) = M * M'  A_ik = Sum(j=1 to n) M_i j * M_k j   i,k = 1, 2, ... , m
	for(i=0;i<m;i++)
		for(k=0;k<m;k++)
		{
			A[i][k] = 0;
			for(j=0;j<n;j++)
				A[i][k] = A[i][k] + M[i][j] * M[k][j];
		}
	
	// diagonalize A with U to hold eigenvectors
	diags(A, U, m, eps);
	
	//put eigenvalues into d
	for(i=0;i<m;i++)
		d[i] = A[i][i];
	
	orderD(d, U, m);  // reorder columns of U so d has eigenvalues in descending order
	
	// copy leading columns of U1 into U up to norm^2 < eps
	
	for(j=0;j<n;j++)
	{
		if(d[j]>=eps)
		{
			for(i=0;i<m;i++)
				U[i][j] = U1[i][j];
		}
	}
	
	for (i=0;i<m;i++)
		free(A[i]);
	free(A);
	
	for (i=0;i<m;i++)
		free(U1[i]);
	free(U1);
	
	free(d);

}/* svd2 */

